/******************************************************************************
 *
 * Dynamixel AX-12 Servo Control Library
 * Filename: ax12.c
 *****************************************************************************/
#include <p18cxxx.h>
#include <usart.h>			//Functions for the on-chip EUSART
#include "ax12.h"			//Prototype declarations for AX12 library

/******************************************************************************
 * AX_SetId()
 *
 * Changes the ID of an AX12 servo to a new value.
 *
 * Parameters:
 *		oldid (char) - the old id # of the AX12 servo
 *		newid (char) - the new id # to which to change to
 * Returns: void
 ******************************************************************************/
void AX_SetId(char oldid, char newid) {
	char parameters[2], paramLength;
	parameters[0] = CT_ID;
	parameters[1] = newid;
	paramLength = 2;
	AX_TxPacket(oldid, I_WRITE_DATA, parameters, paramLength);
} //end AX_SetId()


/******************************************************************************
 * AX_Ping()
 *
 * Pings an AX12 servo for a status packet.
 *
 * Parameters:
 *		id (char) - the id of the AX12 to ping
 * Returns: void
 ******************************************************************************/
void AX_Ping(char id) {
	AX_TxPacket(id, I_PING, 0, 0);
} //end AX_Ping()


/******************************************************************************
 * AX_RxPacket()
 *
 * Receives a packet from an AX12 servo.
 *
 * Parameters: none
 * Returns:
 *		char* - pointer to the retrieved string
 ******************************************************************************/
char* AX_RxPacket() {
	//char length, id, errorByte, checksum, i, curByte;
	char i, length;
	char packet[32];
	packet[0] = AX_RxByte();	//0xFF
	packet[1] = AX_RxByte();	//0xFF
	packet[2] = AX_RxByte();	//length
	packet[3] = AX_RxByte();	//error
	length = packet[3] + 3;
	for (i = 3; i < length; i++) {
		packet[i] = AX_RxByte();
	}
	return packet;
} //end AX_RxPacket()


/******************************************************************************
 * AX_RxByte()
 *
 * Receives a byte from an AX12 servo data line.
 *
 * Parameters: none
 * Returns: (char) the received byte
 ******************************************************************************/
char AX_RxByte() {
	while (DataRdyUSART() == 0) {
		;
	}
	return ReadUSART();
} //end AX_RxByte()


/******************************************************************************
 * AX_RxByteRdy()
 *
 * Determines if a byte from an AX12 servo is ready to be read.
 *
 * Parameters: none
 * Returns: (char) 1 if byte is ready, 0 if not
 ******************************************************************************/
char AX_ByteRdy() {
	return DataRdyUSART();
} //end AX_ByteRdy()


/******************************************************************************
 * AX_TxPacket()
 *
 * Transmits a packet on the AX12 servo data line.
 *
 * Parameters:
 *		id (char) - the ID of the AX12 which should receive the packet
 *		instruction (char) - the instruction type of the packet
 *		parameters (char*) - data array of the packet
 *		paramLength (char) - length of the data array
 * Returns: void
 ******************************************************************************/
void AX_TxPacket(char id, char instruction, char *parameters, char paramLength) {
	char length, i, checksum;
	length = 2 + paramLength;
	checksum = 0;
	AX_SendByte(0xFF);		//required by AX12 protocol to initiate a packet
	AX_SendByte(0xFF);		//required by AX12 protocol to initiate a packet
	AX_SendByte(id);
	checksum = checksum + id;
	AX_SendByte(length);
	checksum = checksum + length;
	AX_SendByte(instruction);
	checksum = checksum + instruction;
	for (i = 0; i < paramLength; i++) {
		AX_SendByte(parameters[i]);	
		checksum = checksum + parameters[i];
	}
	checksum = ~checksum;
	AX_SendByte(checksum);
	while(BusyUSART()) {		//wait for last byte to be sent
	}
} //end AX_TxPacket()


/******************************************************************************
 * AX_SendByte()
 *
 * Sends a single byte onto the AX12 data line.
 *
 * Parameters:
 *		toSend (char) - byte to send onto the data line
 * Returns: void
 ******************************************************************************/
void AX_SendByte(char toSend) {
	while(BusyUSART()) {		//wait for transmit register to be ready
	}
	WriteUSART(toSend);
} //AX_SendByte()


/******************************************************************************
 * AX_SetupUSART()
 *
 * Setup the PIC USART to transmit at 1 MBit/s.
 *
 * Parameters:
 *		id (char) - the ID of the AX12 which should receive the packet
 *		instruction (char) - the instruction type of the packet
 *		parameters (char*) - data array of the packet
 *		paramLength (char) - length of the data array
 * Returns: void
 ******************************************************************************/
void AX_SetupUSART() {
	SPBRG = 2;
	SPBRGH = 0;

	TXSTAbits.SYNC = 0;
	RCSTAbits.SPEN = 1;
	TXSTAbits.TX9 = 0;
	TXSTAbits.SENDB = 0;
	TXSTAbits.BRGH = 1;
	RCSTAbits.RX9 = 0;
	RCSTAbits.CREN = 0;
	BAUDCONbits.BRG16 = 0;
	BAUDCONbits.WUE = 0;
	BAUDCONbits.ABDEN = 0;
	TXSTAbits.TXEN = 1;
} //end AX_SetupUSART()

